<?php

defined('ABSPATH') || exit();

use Automattic\WooCommerce\Blocks\Assets\Api;
use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

require_once PAYREX_GATEWAY_DIR_PATH . 'includes/class-payrex-tokenizer.php';

/**
 * This is the server-side class used to handle server-side logic of the PayRex integration to the merchant's website.
 *
 */
final class PayrexBlock extends AbstractPaymentMethodType
{
    private $gateway;

    /**
     * Payment method name that matches the id from the client-side integration.
     *
     * @var string
     */
    protected $name = "payrex";

    public function initialize()
    {
        $this->settings = get_option('woocommerce_payrex_settings', []);

        $gateways = WC()->payment_gateways->payment_gateways();
        $this->gateway = $gateways[$this->name];
    }

    /**
     * Extended from the parent class.
     */
    public function is_active()
    {
        return filter_var($this->get_setting('enabled', false), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns an array of scripts/handles to be registered for this payment method. Extended from the parent class.
     *
     * @return array
     */
    public function get_payment_method_script_handles()
    {
        /**
         * PayRex Javascript Library
         * This plugin connects to PayRex's Public API through this JS library to send payment information from the customers who completes the payment through the woocommerce store.
         * It sends the payment information of the customer to PayRex Public API to process the online payment.
         * This service is provided by PayRex.
         * Terms of use: https://www.payrex.com/terms
         * Privacy Policy: https://www.payrex.com/privacy
         */
        wp_register_script(
            'payrex',
            'https://js.payrexhq.com',
            [],
            time(),
            true
        );

        wp_register_script(
            'payrex-instance',
            plugin_dir_url(PAYREX_GATEWAY_FILE) . 'assets/payrex.js',
            ['payrex'],
            time(),
            true
        );

        $publishableKey = $this->gateway->getPublishableKey();

        wp_localize_script('payrex-instance', 'payrex_gateway_data', ['publishableKey' => $publishableKey]);

        wp_register_script(
            'wc-payment-method-payrex',
            plugin_dir_url(PAYREX_GATEWAY_FILE) . 'build/index.js',
            [
                'wc-blocks-registry',
                'wc-settings',
                'wp-element',
                'wp-html-entities',
                'wp-i18n',
                'payrex',
                'payrex-instance'
            ],
            time(),
            true
        );

        wp_localize_script('wc-payment-method-payrex', 'payrex_payment_method_data', [
            'token' => Payrex_Tokenizer::generate_create_payment_intent_token(),
            'isTestMode' => !$this->gateway->isLivemode()
        ]);

        return ['payrex', 'payrex-instance', 'wc-payment-method-payrex'];
    }

    /**
     * Returns an array of key=>value pairs of data made available to the payment methods script. Extended from the parent class.
     *
     * @return array
     */
    public function get_payment_method_data()
    {
        return [
            'title'                    => $this->gateway->title,
            'description'              => $this->gateway->description,
            'payment_method_icons'     => $this->gateway->getEnabledPaymentMethodIcons(),
            'icon' => $this->gateway->icon
        ];
    }
}
