<?php

defined( 'ABSPATH' ) || exit;
define('PAYREX_CONNECT_BASE_URL', 'https://connect.payrexhq.com');
define('PAYREX_CLIENT_ID', 'clt_YNn71JXPp9mhoqcYcWUop6i4Z42otJcJ');

/**
 * Core class that represents payrex and its woocommerce settings. 
 * It is required to extend WC_Payment_Gateway.
 */

class WC_Gateway_PayRex extends WC_Payment_Gateway
{
    private const CONNECT_BASE_URL = PAYREX_CONNECT_BASE_URL;
    private const CONNECT_CLIENT_ID = PAYREX_CLIENT_ID;
    protected const PAYMENT_METHOD_ICONS = [
        'amex' => 'American Express',
        'diners' => 'Diners Club',
        'discover' => 'Discover',
        'gcash' => 'GCash',
        'jcb' => 'JCB',
        'mastercard' => 'Mastercard',
        'maya' => 'Maya',
        'qrph' => 'QRPH',
        'visa' => 'Visa'
    ];

    public function __construct()
    {
        $description = 'Secure and unified payments powered by PayRex.';

        /**
         * The class attributes here are required attributes of the parent class.
         */
        $this->id = 'payrex';
        $this->icon = plugins_url('assets/payrex-logo-head.svg', PAYREX_GATEWAY_FILE);
        $this->has_fields = true;
        $this->method_title = 'PayRex';
        $this->method_description = $description;
        $this->title = 'Payments via PayRex';
        $this->description = $description;
        $this->enabled = $this->get_option('enabled');
        $this->order_button_text = 'Continue';
        $this->supports = ['products', 'refunds', 'checkout', 'order-pay', 'custom_block_handling'];
        $this->init_form_fields();
        // This is a parent class method used to load the $this->get_option() values.
        $this->init_settings();

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
    }

    /**
     * Returns a boolean attribute if the payment settings is live or test.
     */
    public function isLivemode()
    {
        return $this->get_option('mode') === 'live';
    }

    public function getEnabledPaymentMethodIcons()
    {
        return $this->get_option('enabled_icons', []);
    }

    /**
     * Returns the icon of the payment gateway. Extended from the parent class.
     */
    public function get_icon()
    {
        $icon_url = $this->icon;
        $logo_size = '28px';

        return "<img src='$icon_url' alt='PayRex Logo' style='width:$logo_size; height: auto;' />";
    }

    /**
     * Initializes the form fields for the payment gateway settings. Extended from the parent class.
     */
    public function init_form_fields()
    {
        if ($this->isAccountConnected()) {
            $button_label = 'Disconnect Account';
            $button_url   = esc_url($this->getDisconnectUrl());
            $button_style = 'button-secondary';
            $hideFields = false;
        } else {
            $button_label = 'Connect Account';
            $button_url   = esc_url($this->getConnectUrl());
            $button_style = 'button-primary';
            $hideFields = true;
        }

        $form_fields = [
            'enabled'            => [
                'title'       => 'Enable/Disable',
                'label'       => 'Enable PayRex',
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'no',
                'desc_tip'    => true,
            ],
            'account_action' => [
                'title'       => 'Merchant Account Connection',
                'type'        => 'title',
                'description' => "<a href='{$button_url}' class='button {$button_style}'>{$button_label}</a>",
            ]
        ];

        if (!$hideFields) {
            $form_fields = array_merge(
                $form_fields,
                [
                    'mode' => [
                        'title'       => 'Mode',
                        'type'        => 'select',
                        'description' => 'Choose between test mode or live mode.',
                        'default'     => 'test',
                        'desc_tip'    => true,
                        'options'     => [
                            'live' => 'Live Mode',
                            'test'    => 'Test Mode',
                        ],
                    ],
                    'trade_name' => [
                        'title'       => 'Trade Name',
                        'type'        => 'text',
                        'description' => "Your PayRex merchant's trade name.",
                        'default'     => '',
                        'desc_tip'    => true,
                        'custom_attributes' => [
                            'readonly' => 'readonly',
                        ]
                    ],
                    'enabled_icons' => [
                        'title'       => 'Enabled Payment Method Icons',
                        'type'        => 'multiselect',
                        'description' => 'Choose which payment method icons to show at checkout.',
                        'default'     => '',
                        'desc_tip'    => true,
                        'options'     => $this::PAYMENT_METHOD_ICONS
                    ]
                ]
            );

            $webhook_field = $this->isLivemode() ? 'live_webhook_id' : 'test_webhook_id';

            $mode = $this->get_option('mode');

            if ($this->isPayrexWebhookConfigured()) {
                $webhook_url = home_url("/wp-json/wc-payrex/v1/webhook?mode={$mode}");

                $form_fields = array_merge(
                    $form_fields,
                    [
                        $webhook_field => [
                            'title'       => 'Webhook ID',
                            'type'        => 'text',
                            'description' => $webhook_url,
                            'default'     => '',
                            'custom_attributes' => [
                                'readonly' => 'readonly',
                            ]
                        ]
                    ]
                );
            } else {
                $setup_webhook_url = $this->getSetupWebhookUrl($mode);

                $form_fields = array_merge(
                    $form_fields,
                    [
                        'webhook_action' => [
                            'title'       => 'Webhook',
                            'type'        => 'title',
                            'description' => "<a href='{$setup_webhook_url}' class='button button-secondary'>Setup Webhook</a>",
                        ]
                    ]
                );
            }
        }

        // This is required to set the form fields in the parent class.
        $this->form_fields = $form_fields;
    }

    /**
     * Checks if the extension is ready to use. Extended from the parent class.
     */
    public function is_available()
    {
        if (get_woocommerce_currency() !== 'PHP') {
            return false;
        }

        if (!$this->isPayrexWebhookConfigured()) {
            return false;
        }

        return parent::is_available();
    }

    public function getSecretKey()
    {
        $mode = $this->get_option('mode');

        $apiKeyIndex = $mode === 'live' ? "live_secret_key" : "test_secret_key";

        return $this->get_option($apiKeyIndex);
    }

    public function getPublishableKey()
    {
        $mode = $this->get_option('mode');

        $apiKeyIndex = $mode === 'live' ? "live_publishable_key" : "test_publishable_key";

        return $this->get_option($apiKeyIndex);
    }

    /**
     * Triggered when a customer will start to pay. Extended from the parent class.
     *
     * @param int $order_id Order ID.
     * @return array
     */
    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);

        $secretKey = $this->getSecretKey();
        
        $paymentIntentId = $order->get_meta('_payrex_intent_id');

        $client = new \Payrex\PayrexClient($secretKey);

        $paymentIntent = null;

        $orderTotalAmount = (int) round($order->get_total() * 100);

        if (!empty($paymentIntentId)) {
            try {
                $paymentIntent = $client->paymentIntents->retrieve($paymentIntentId);

                if (!in_array($paymentIntent->status, ['awaiting_payment_method', 'awaiting_next_action'])) {
                    $intent =  null;
                }
            } catch (\Payrex\Exceptions\ResourceNotFoundException $e) {
                $paymentIntent = null;
            } catch (\Payrex\Exceptions\BaseException $e) {
                $paymentIntent = null;
            }
        }

        if (!$paymentIntent) {
            try {
                $paymentIntent = $client->paymentIntents->create([
                    'amount' => $orderTotalAmount,
                    'currency' => get_woocommerce_currency(),
                    'metadata' => [
                        'order_id' => $order->get_id()
                    ]
                ]);

                $order->update_meta_data('_payrex_intent_id', sanitize_text_field($paymentIntent->id));
                $order->save();
            } catch (\Payrex\Exceptions\BaseException $e) {
                return new WP_Error('payment-intent-error', $e->getError(), ['status' => 400]);
            }
        }

        if ($paymentIntent && $paymentIntent->amount != $orderTotalAmount) {
            try {
                $httpClient = (new Payrex\Services\BaseService($client))->httpClient;

                $response = $httpClient->request([
                    'method' => 'PUT',
                    'url'    => "{$client->apiBaseUrl}/payment_intents/{$paymentIntent->id}",
                    'params' => [
                        'amount' => $orderTotalAmount
                    ]
                ]);

                $intent = new \Payrex\Entities\PaymentIntent($response);

                $order->update_meta_data('_payrex_intent_id', sanitize_text_field($paymentIntent->id));
                $order->save();
            } catch (\Payrex\Exceptions\BaseException $e) {
                return new WP_Error('payment-intent-error', $e->getError(), ['status' => 400]);
            }
        }

        $url = home_url('/wp-json/wc-payrex/v1/payment-callback?order_id=' . $order_id . '&order_key=' . $order->get_order_key() . '&source=checkout');

        return [
            'result'   => 'success',
            'redirect' => $url,
            'payment_intent_id' => $paymentIntent->id,
            'order_id' => $order_id,
            'order_key' => $order->get_order_key(),
            'client_secret' => $paymentIntent->client_secret
        ];
    }

    /**
     * Process the refund of an order. Extended from the parent class.
     */
    public function process_refund($order_id, $amount = null, $reason = '')
    {
        $order = wc_get_order($order_id);

        $refunds = $order->get_refunds();

        $refund = end($refunds);

        $refund_amount = $refund->get_amount();
        $refund_reason = $refund->get_reason();

        $transaction_id = $order->get_transaction_id();

        if (!$transaction_id || empty($transaction_id)) {
            return new WP_Error('missing_transaction_id', 'Order does not have a PayRex payment reference ID.');
        }

        if ($refund_amount != $amount || $refund_reason != $reason) {
            return new WP_Error('invalid_request', 'Mismatch refund record.');
        }

        try {
            $secret_key = $this->getSecretKey();

            $payrex_client = new \Payrex\PayrexClient($secret_key);

            $refund_data = [
                'amount'     => round($amount * 100),
                'currency'   => get_woocommerce_currency(),
                'payment_id' => $transaction_id,
                'reason'     => 'others'
            ];

            if (!empty($reason)) {
                if (strlen($reason) > 50) {
                    $other_reason = substr($reason, 0, 50);
                } else {
                    $other_reason = $reason;
                }

                $refund_data['other_reason'] = $other_reason;
            }

            $payrex_refund = $payrex_client->refunds->create($refund_data);

            $refund_status = $payrex_refund->status;

            if ($refund_status === 'succeeded') {
                $refund->update_meta_data('_refund_pending', 'no');
                $refund->update_meta_data('_payrex_refund_id', $payrex_refund->id);
                $refund->save();

                $order->add_order_note("Refunded {$amount} via PayRex: {$payrex_refund->id}. Reason: {$refund->reason}");
            } elseif ($refund_status === 'pending') {
                $refund->update_meta_data('_refund_pending', 'yes');
                $refund->update_meta_data('_payrex_refund_id', $payrex_refund->id);
                $refund->save();

                $order->add_order_note("Refund of {$amount} initiated. Waiting for confirmation from PayRex: {$payrex_refund->id}.");
            } elseif ($refund_status === 'failed') {
                return new WP_Error('refund_error', 'Failed refunding payment.');
            }

            return true;
        } catch (\Payrex\Exceptions\BaseException $e) {
            $order->add_order_note("Refund of {$amount} failed. PayRex encountered an error: {$e->getError()[0]->detail}.");
            return new WP_Error('refund_error', $e->getError()[0]->detail);
        }
    }

    /**
     * Default payment display of the extension. Extended from the parent class.
     */
    // phpcs:disable WordPress.Security.NonceVerification.Recommended
    public function payment_fields()
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag
        $order_id = isset($_GET['key']) ? wc_get_order_id_by_order_key(sanitize_text_field(wp_unslash($_GET['key']))) : null;

        $paymentIntent = null;

        try {
            $order = wc_get_order($order_id);
            $paymentIntentId = $order->get_meta('_payrex_intent_id');

            $secretKey = $this->getSecretKey();

            $client = new \Payrex\PayrexClient($secretKey);
            $paymentIntent = $client->paymentIntents->retrieve($paymentIntentId);

            if (!in_array($paymentIntent->status, ['awaiting_payment_method', 'awaiting_next_action'])) {
                $paymentIntent =  null;
            }
        } catch (\Payrex\Exceptions\ResourceNotFoundException $e) {
            $paymentIntent = null;
        } catch (\Payrex\Exceptions\BaseException $e) {
            $paymentIntent = null;
        }

        $orderTotalAmount = (int) round($order->get_total() * 100);

        if (!$paymentIntent) {
            try {
                $intent = $client->paymentIntents->create([
                    'amount' => $orderTotalAmount,
                    'currency' => get_woocommerce_currency(),
                    'metadata' => [
                        'order_id' => $order->get_id()
                    ]
                ]);

                $order->update_meta_data('_payrex_intent_id', sanitize_text_field($paymentIntent->id));
                $order->save();
            } catch (\Payrex\Exceptions\BaseException $e) {
                return new WP_Error('payment-intent-error', $e->getError(), ['status' => 400]);
            }
        }

        if ($paymentIntent && $paymentIntent->amount !== $orderTotalAmount) {
            try {
                $httpClient = (new Payrex\Services\BaseService($client))->httpClient;

                $response = $httpClient->request([
                    'method' => 'PUT',
                    'url'    => "{$client->apiBaseUrl}/payment_intents/{$paymentIntent->id}",
                    'params' => [
                        'amount' => $orderTotalAmount
                    ]
                ]);

                $paymentIntent = new \Payrex\Entities\PaymentIntent($response);

                $order->update_meta_data('_payrex_intent_id', sanitize_text_field($paymentIntent->id));
                $order->save();
            } catch (\Payrex\Exceptions\BaseException $e) {
                return new WP_Error('payment-intent-error', $e->getError(), ['status' => 400]);
            }
        }

        // Used if a user is logged in to complete a payment
        wc_get_template(
            'payment-method.php',
            [
                'gateway_id' => $this->id,
                'chosen' => $this->chosen,
                'order_button_text' => $this->order_button_text,
                'client_secret' => $paymentIntent->client_secret,
                'redirect_url' => $url = home_url('/wp-json/wc-payrex/v1/payment-callback?order_id=' . $order_id . '&order_key=' . $order->get_order_key() . '&source=order-pay')
            ],
            '',
            PAYREX_GATEWAY_DIR_PATH . 'templates/checkout/'
        );

        echo '
            <div style="background-color:white;">
                <div id="payrex-payment-element" />
            </div>
        ';
    }
    // phpcs:enable WordPress.Security.NonceVerification.Recommended

    /**
     * Outputs the payment gateway settings screen. Extended from the parent class.
     */
    public function admin_options()
    {
        // $this->generate_settings_html is already pre-escaped.
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        echo '<table class="form-table">' . $this->generate_settings_html($this->get_form_fields(), false) . '</table>';

        if (get_woocommerce_currency() !== 'PHP') {
            $woocommerce_settings_url = admin_url('admin.php?page=wc-settings&tab=general');

            echo '<div class="notice notice-error"><p><strong>PayRex only supports PHP currency. Please update your <a href="' . esc_url($woocommerce_settings_url) . '" />currency settings</a> to use PayRex.</strong></p></div>';
        }

        if (!is_ssl() && $this->isLivemode()) {
            echo '<div class="notice notice-error"><p><strong>SSL certificate is not detected. Your checkout may not be secure! Please ensure your server has a valid SSL certificate.</strong></p></div>';
        }
    }

    private function getConnectUrl()
    {
        $client_id    = $this::CONNECT_CLIENT_ID;
        $return_url = html_entity_decode(wp_nonce_url(
            admin_url('admin-post.php?action=payrex_connect'),
            'payrex_connect'
        ));
        $nonce = wp_create_nonce('payrex_connect');

        $url = $this::CONNECT_BASE_URL . "/oauth/authorization";
        $params = [
            'clientId'     => $client_id,
            'returnUrl'  => $return_url,
            'nonce'         => $nonce,
        ];

        return $url . '?' . http_build_query($params);
    }

    private function getDisconnectUrl()
    {
        return wp_nonce_url(
            admin_url('admin-post.php?action=payrex_disconnect'),
            'payrex_disconnect'
        );
    }

    private function getSetupWebhookUrl($mode)
    {
        return wp_nonce_url(
            admin_url("admin-post.php?action=payrex_setup_webhook&mode={$mode}"),
            'payrex_setup_webhook'
        );
    }

    private function isPayrexWebhookConfigured()
    {
        $webhookField = $this->isLivemode() ? 'live_webhook_id' : 'test_webhook_id';

        return !empty($this->get_option($webhookField));
    }

    private function isAccountConnected()
    {
        return !empty($this->getSecretKey());
    }
}
