<?php

defined('ABSPATH') || exit();

class Payrex_Controller_Payment_Intent
{
    /**
     * Method to register the routes to WooCommerce
     */
    public function register_routes()
    {
        register_rest_route(
            'wc-payrex/v1',
            'payment-intents',
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'create_payment_intent_from_order'],
                'permission_callback' => '__return_true',
                'args'                => [
                    'order_id'          => ['required' => true],
                    'order_key'         => ['required' => true]
                ],
            ]
        );
    }

    /**
     * @param \WP_REST_Request $request
     */
    public function create_payment_intent_from_order($request)
    {
        $gateway = WC()->payment_gateways()->get_available_payment_gateways()['payrex'];

        if (!$gateway) {
            return new WP_Error('action_forbidden', 'Action forbidden.', ['status' => 403]);
        }

        $headers = getallheaders();
        $token   = isset($headers['X-Wc-Payrex-Token']) ? $headers['X-Wc-Payrex-Token'] : null;

        if (!$token || !Payrex_Tokenizer::validate_create_payment_intent_token($token)) {
            return new WP_Error('action_forbidden', 'Action forbidden.', ['status' => 403]);
        }

        $orderId = $request['order_id'];
        $orderKey = $request['order_key'];
        $order = wc_get_order(absint($orderId));

        if (!$order || !hash_equals($order->get_order_key(), $orderKey)) {
            return new WP_Error('resource_not_found', 'Order not found.', ['status' => 404]);
        }

        $current_user_id = get_current_user_id();

        if ($current_user_id && $order->get_user_id() !== $current_user_id) {
            return new WP_Error('action_forbidden', 'Action forbidden.', ['status' => 403]);
        }

        if ($order->is_paid()) {
            return new WP_Error('invalid_status', 'Order is paid.');
        }

        $secretKey = $gateway->getSecretKey();

        $paymentIntentId = $order->get_meta('_payrex_intent_id');

        $client = new \Payrex\PayrexClient($secretKey);

        $orderTotalAmount = (int) round($order->get_total() * 100);
        $currency = get_woocommerce_currency();

        $paymentIntent = null;

        if ($paymentIntentId || !empty($paymentIntentId)) {
            try {
                $paymentIntent = $client->paymentIntents->retrieve($paymentIntentId);

                if (!in_array($paymentIntent->status, ['awaiting_payment_method', 'awaiting_next_action'])) {
                    return new WP_Error('invalid_status', 'Payment intent is invalid');
                }
            } catch (\Payrex\Exceptions\ResourceNotFoundException $e) {
                $paymentIntent = null;
            } catch (\Payrex\Exceptions\BaseException $e) {
                $paymentIntent = null;
            }
        }

        if (!$paymentIntent) {
            try {
                $paymentIntent = $client->paymentIntents->create([
                    'amount' => $orderTotalAmount,
                    'currency' => $currency,
                    'metadata' => [
                        'order_id' => $orderId
                    ]
                ]);
            } catch (\Payrex\Exceptions\BaseException $e) {
                return new WP_Error('payment_intent_error', $e->getError(), ['status' => 400]);
            }
        }

        if ($paymentIntent->amount != $orderTotalAmount) {
            try {
                $httpClient = (new Payrex\Services\BaseService($client))->httpClient;

                $response = $httpClient->request([
                    'method' => 'PUT',
                    'url'    => "{$client->apiBaseUrl}/payment_intents/{$paymentIntent->id}",
                    'params' => [
                        'amount' => $orderTotalAmount
                    ]
                ]);

                $paymentIntent = new \Payrex\Entities\PaymentIntent($response);
            } catch (\Payrex\Exceptions\BaseException $e) {
                return new WP_Error('payment_intent_error', $e->getError(), ['status' => 400]);
            }
        }

        $order->update_meta_data('_payrex_intent_id', $paymentIntent->id);
        $order->save();

        return rest_ensure_response([
            'client_secret' => $paymentIntent->client_secret
        ]);
    }
}
