import { useEffect } from 'react'
import { usePaymentIntent } from '../usePaymentIntent'
import { usePayrexSession } from '../usePayrexSession'
import PaymentMethodSkeleton from './PaymentMethodSkeleton'
import TestModeBanner from './TestModeBanner'

const isTestMode = [1, '1'].includes(window.payrex_payment_method_data?.isTestMode)

// Used if a user is not logged in to complete a payment
const PaymentMethod = (paymentMethodProps) => {
  const {
    billing,
    emitResponse,
    eventRegistration,
  } = paymentMethodProps

  const { setPayrexSession, getPayrexSession } = usePayrexSession()
  const { onCheckoutSuccess, onCheckoutValidation } = eventRegistration
  const { loading, fetchPaymentIntent } = usePaymentIntent()

  const mountPaymentElement = async (clientSecret) => {
    if (clientSecret) {
      const elements = window.payrex.elements({
        clientSecret: clientSecret
      })

      const paymentElement = elements.create('payment', {
        fields: {
          billingDetails: {
            email: 'never',
            name: 'never',
            phone: 'never',
            address: {
              line1: 'never',
              line2: 'never',
              city: 'never',
              postalCode: 'never',
              state: 'never',
              country: 'never'
            }
          }
        }
      })

      paymentElement.mount('#payrex-payment-element')

      setPayrexSession('elements', elements)

      return elements
    }
  }

  const setupPaymentElement = async () => {
    let clientSecret = getPayrexSession('payment_intent_id')

    if (!clientSecret) {
      const paymentIntent = await fetchPaymentIntent()

      clientSecret = paymentIntent.client_secret

      setPayrexSession('payment_intent_id', clientSecret)
    }

    await mountPaymentElement(clientSecret)
  }

  useEffect(() => {
    setupPaymentElement()
  }, [])

  useEffect(() => {
    const unsubscribe = onCheckoutValidation(async (_parameters) => {
      const paymentNoticeWrapper =
        '#payment-method .wc-block-components-notices'
      const notices = document.querySelector(paymentNoticeWrapper)
      Array.from(notices.children).forEach((child) => {
        if (
          child.dataset.contexts &&
          child.dataset.contexts === 'wc/checkout/payments'
        ) {
          notices.removeChild(child)
        }
      })
    })

    return () => {
      unsubscribe()
    }
  }, [onCheckoutValidation])

  useEffect(() => {
    const unsubscribe = onCheckoutSuccess(async (successParameters) => {
      let mountedPaymentElement = getPayrexSession('elements')

      const clientSecret = getPayrexSession('payment_intent_id')
      const orderClientSecret =
        successParameters.processingResponse.paymentDetails.client_secret

      if (clientSecret !== orderClientSecret) {
        setPayrexSession('payment_intent_id', orderClientSecret)

        mountedPaymentElement = await mountPaymentElement(orderClientSecret)

        return {
          type: emitResponse.responseTypes.ERROR,
          message:
            'Order details have changed. Please re-enter your payment information.',
          messageContext: emitResponse.noticeContexts.PAYMENTS
        }
      }

      const { redirectUrl } = successParameters

      const attachResponse = await window.payrex.attachPaymentMethod({
        elements: mountedPaymentElement,
        options: {
          return_url: redirectUrl
        },
        attachParams: {
          paymentMethodData: {
            billingDetails: {
              name: `${billing.billingData.first_name} ${billing.billingData.last_name}`,
              email: billing.billingData.email,
              address: {
                line1: billing.billingData.address_1,
                city: billing.billingData.city,
                state: billing.billingData.state,
                postalCode: billing.billingData.postal_code,
                country: billing.billingData.country
              },
              ...(billing.billingData.phone === ''
                ? {}
                : { phone: billing.billingData.phone })
            }
          }
        }
      })

      const successfulStatus = ['awaiting_capture', 'succeeded']

      const failedStatus = [
        'awaiting_action',
        'awaiting_payment_method',
        'canceled'
      ]

      if (typeof attachResponse === 'string') {
        return {
          type: emitResponse.responseTypes.ERROR,
          message: attachResponse,
          messageContext: emitResponse.noticeContexts.PAYMENTS
        }
      } else if (!attachResponse) {
        return {
          type: emitResponse.responseTypes.ERROR,
          message: 'The latest payment authentication has failed or cancelled.',
          messageContext: emitResponse.noticeContexts.PAYMENTS
        }
      } else if (successfulStatus.includes(attachResponse.status)) {
        return {
          type: emitResponse.responseTypes.SUCCESS,
          redirectUrl: orderReceivedUrl
        }
      } else if (failedStatus.includes(attachResponse.status)) {
        const errorMessage =
          attachResponse.last_payment_error?.detail ||
          'The latest payment authentication has failed or cancelled.'

        return {
          type: emitResponse.responseTypes.ERROR,
          message: errorMessage,
          messageContext: emitResponse.noticeContexts.PAYMENTS
        }
      } else {
        return {
          type: emitResponse.responseTypes.ERROR,
          message: 'The latest payment authentication has failed or cancelled.',
          messageContext: emitResponse.noticeContexts.PAYMENTS
        }
      }
    })

    return () => {
      unsubscribe()
    }
  }, [
    billing,
    emitResponse.responseTypes.ERROR,
    emitResponse.responseTypes.SUCCESS,
    onCheckoutSuccess
  ])

  return (
    <div>
      {loading}
      {isTestMode && <TestModeBanner />}
      {loading && <PaymentMethodSkeleton />}
      <div id="payrex-payment-element" />
    </div>
  )
}

export default PaymentMethod
